<?php

/**
 * PowerHour\Utility\Comments\Component class
 *
 * @package power_hour
 */

namespace PowerHour\Utility\Common;

use PowerHour\Utility\Component_Interface;
use PowerHour\Utility\Templating_Component_Interface;
use function add_action;

/**
 * Class for managing comments UI.
 *
 * Exposes template tags:
 * * `power_hour()->the_comments( array $args = array() )`
 *
 * @link https://wordpress.org/plugins/amp/
 */
class Component implements Component_Interface, Templating_Component_Interface
{
	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'common';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{
		add_filter('widget_tag_cloud_args', array($this, 'powerhour_widget_tag_cloud_args'), 100);
		add_filter('wp_list_categories', array($this, 'powerhour_categories_postcount_filter'), 100);
		add_filter('get_archives_link', array($this, 'powerhour_style_the_archive_count'), 100);
		add_filter('upload_mimes', array($this, 'powerhour_mime_types'), 100);
		add_action('wp_enqueue_scripts', array($this, 'powerhour_remove_wp_block_library_css'), 100);
		add_filter('pre_get_posts', array($this, 'powerhour_searchfilter'), 100);
		add_theme_support('post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
			'gallery',
			'audio',
		));

		// Open close container
		add_filter('powerhour_container_open', array($this, 'powerhour_add_container_open_if_redux'));
        add_filter('powerhour_container_close', array($this, 'powerhour_add_container_close_if_redux'));

	}

	public function __construct()
	{
		add_filter('the_content', array($this, 'powerhour_remove_empty_p'));
		add_filter('get_the_content', array($this, 'powerhour_remove_empty_p'));
		add_filter('get_the_excerpt', array($this, 'powerhour_remove_empty_p'));
		add_filter('the_excerpt', array($this, 'powerhour_remove_empty_p'));
		add_filter('body_class', array($this, 'powerhour_add_body_classes'));
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `power_hour()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */
	public function template_tags(): array
	{
		return array(
			'iqonic_blog_portfolio_time_link'  => array($this, 'iqonic_blog_portfolio_time_link'),
			'powerhour_pagination' => array($this, 'powerhour_pagination'),
			'powerhour_inner_breadcrumb' => array($this, 'powerhour_inner_breadcrumb'),
			'powerhour_get_embed_video'  => array($this, 'powerhour_get_embed_video'),
			'powerhour_singleportfolio' 	=> array($this, 'powerhour_singleportfolio'),
			'powerhour_owlcarousal'  => array($this, 'powerhour_owlcarousal'),
			'powerhour_scrollsticky'  => array($this, 'powerhour_scrollsticky'),
			'powerhour_masonary'  => array($this, 'powerhour_masonary'),
			

			//post type layouts
			'call_defaultstyle' => array($this, 'call_defaultstyle'),
			'call_titlefirst' => array($this, 'call_titlefirst'),
			'call_imagefirst' => array($this, 'call_imagefirst'),
		);
	}

	
		/**
		 * Gets a nicely formatted string for the published date.
		 */
		public function iqonic_blog_portfolio_time_link()
{


    $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';

    $time_string = sprintf(
        $time_string,
        get_the_date(DATE_W3C),
        get_the_date(),
        get_the_modified_date(DATE_W3C),
        get_the_modified_date()
    );

    $archive_year  = get_the_time('Y');
    $archive_month = get_the_time('m');
    $archive_day   = get_the_time('d');


    echo sprintf("%s", $time_string);
}
	

	public function powerhour_scrollsticky()
	{
		wp_enqueue_script('scrollsticky', get_template_directory_uri() . '/assets/js/scroll-sticky.min.js', array('jquery') , '1.0' , true);
	}

	public function powerhour_singleportfolio()
	{
		wp_enqueue_style('single_portfolio', get_template_directory_uri() . '/assets/css/src/single-portfolio.css', array(), '1.0', "all");
		wp_enqueue_script('magnific-min', get_template_directory_uri() . '/assets/js/singleportfolio.min.js', array(), '1.0' , true);
	}

	public function powerhour_owlcarousal()
	{
		wp_enqueue_style('single_portfolio-owl', get_template_directory_uri() . '/assets/css/vendor/owl.carousel.min.css', array(), '1.0', "all");
		wp_enqueue_script('owl-carousel-general', get_template_directory_uri() . '/assets/js/owl-general.min.js', array(), '1.0', true);
		wp_enqueue_script('owl-carousel-min', get_template_directory_uri() . '/assets/js/vendor/owl.carousel.min.js', array(), '1.0', true);
	}

	public function powerhour_masonary()	
	{
		wp_enqueue_style('single-portfolio', get_template_directory_uri() . '/assets/css/portfolio.min.css', array(), '1.0', "all");
		wp_enqueue_script('single-masonry-min', get_template_directory_uri() . '/assets/js/vendor/masonry.pkgd.min.js', array(), '1.0', true);
		wp_enqueue_script('imageload', get_template_directory_uri() . '/assets/js/vendor/imagesloaded.pkgd.min.js', array(), '1.0', true);
	}

	public function call_defaultstyle()
	{
		if (!is_search()) {
			if (isset($powerhour_options['display_featured_image'])) {
				$options = $powerhour_options['display_featured_image'];
				if ($options == "yes") {
					get_template_part('template-parts/content/entry_thumbnail', get_post_type());
				}
			} else {
				get_template_part('template-parts/content/entry_thumbnail', get_post_type());
			}
		} ?>
		<div class="powerhour-blog-detail">
		<?php
		get_template_part('template-parts/content/entry_meta', get_post_type());
		get_template_part('template-parts/content/entry_title', get_post_type());
	}

	public function call_titlefirst() { ?>
		<div class="powerhour-blog-detail single_type-1"> <?php
			get_template_part('template-parts/content/entry_title', get_post_type());
			get_template_part('template-parts/content/entry_meta', get_post_type());

			if (!is_search()) {
				if (isset($powerhour_options['display_featured_image'])) {
					$options = $powerhour_options['display_featured_image'];
					if ($options == "yes") {
						get_template_part('template-parts/content/entry_thumbnail', get_post_type());
					}
				} else {
					get_template_part('template-parts/content/entry_thumbnail', get_post_type());
				}
			}
		echo '</div>';
	}

	public function call_imagefirst() {
		$page_id = get_queried_object_id();
		$id = '';
		if (function_exists('get_field')) {
			$id = get_field('acf_blog_detail_style', $page_id);
		}

		$powerhour_options = get_option('power-hour-options');
		if (!is_search()) {
			if (isset($powerhour_options['display_featured_image'])) {
				$options = $powerhour_options['display_featured_image'];
				if ($options == "yes") {
					get_template_part('template-parts/content/entry_thumbnail', get_post_type());
				}
			} else {
				get_template_part('template-parts/content/entry_thumbnail', get_post_type());
			}
		} ?>

		<div class="powerhour-blog-detail single_type-2 ">
			<?php
			if ($id == 2) {
				get_template_part('template-parts/content/entry_title', get_post_type());
				get_template_part('template-parts/content/entry_meta', get_post_type());
			} elseif ($powerhour_options['blog_detail_style'] == '2') {
				get_template_part('template-parts/content/entry_title', get_post_type());
				get_template_part('template-parts/content/entry_meta', get_post_type());
			}
		echo '</div>';
	}

	function powerhour_add_body_classes($classes)
	{
		if (class_exists('ReduxFramework')) {
			$powerhour_option = get_option('power-hour-options');
			$id = get_queried_object_id();
			
			$acf_display_header = get_post_meta($id,'display_header',true);
			if($acf_display_header == 'no') return $classes;

			$page_header_layout = (function_exists('get_field') && !empty($id)) ? get_post_meta($id, 'header_layout_type', true) : '';
			$page_header_on = (function_exists('get_field') && !empty($id)) ? get_post_meta($id, 'display_header', true) : '';

			if($page_header_layout == 'custom' && $page_header_on == 'yes'){
				$classes = array_merge($classes, array('iqonic-custom-header'));
			} elseif ($powerhour_option['header_layout'] == 'custom' ) {
				$classes = array_merge($classes, array('iqonic-custom-header'));
			} else {
				$classes = array_merge($classes, array('iqonic-default-header'));
			}
		} else {
			$classes = array_merge($classes, array('iqonic-default-header'));
		}

		return $classes;
	}


	function powerhour_get_embed_video($post_id)
	{
		$post = get_post($post_id);
		$content = do_shortcode(apply_filters('the_content', $post->post_content));
		$embeds = get_media_embedded_in_content($content);
		if (!empty($embeds)) {
			foreach ($embeds as $embed) {
				if (strpos($embed, 'video') || strpos($embed, 'youtube') || strpos($embed, 'vimeo') || strpos($embed, 'dailymotion') || strpos($embed, 'vine') || strpos($embed, 'wordPress.tv') || strpos($embed, 'embed') || strpos($embed, 'audio') || strpos($embed, 'iframe') || strpos($embed, 'object')) {
					return $embed;
				}
			}
		} else {
			return;
		}
	}

	function powerhour_remove_empty_p($string)
	{
		return preg_replace('/<p>(?:\s|&nbsp;)*?<\/p>/i', '', $string);
	}

	function powerhour_remove_wp_block_library_css()
	{
		wp_dequeue_style('wp-block-library-theme');
	}

	public function powerhour_widget_tag_cloud_args($args)
	{
		$args['largest'] = 1;
		$args['smallest'] = 1;
		$args['unit'] = 'em';
		$args['format'] = 'list';

		return $args;
	}
	function powerhour_mime_types($mimes)
	{
		$mimes['svg'] = 'image/svg+xml';
		return $mimes;
	}
	function powerhour_categories_postcount_filter($variable)
	{
		$variable = str_replace('(', '<span class="archiveCount"> (', $variable);
		$variable = str_replace(')', ') </span>', $variable);
		return $variable;
	}

	function powerhour_style_the_archive_count($links)
	{
		$links = str_replace('</a>&nbsp;(', '</a> <span class="achiveCount"> (', $links);
		$links = str_replace('&nbsp;)</li>', ' </li></span>', $links);
		return $links;
	}

	public function powerhour_pagination($numpages = '', $pagerange = '', $paged = '')
	{

		if (empty($pagerange)) {
			$pagerange = 2;
		}
		global $paged;
		if (empty($paged)) {
			$paged = 1;
		}
		if ($numpages == '') {
			global $wp_query;
			$numpages = $wp_query->max_num_pages;
			if (!$numpages) {
				$numpages = 1;
			}
		}
		/**
		 * We construct the pagination arguments to enter into our paginate_links
		 * function.
		 */
		$pagination_args = array(
			'format' => '?paged=%#%',
			'total' => $numpages,
			'current' => $paged,
			'show_all' => false,
			'end_size' => 1,
			'mid_size' => $pagerange,
			'prev_next' => true,
			'prev_text'       => '<i class="fas fa-chevron-left"></i>',
			'next_text'       => '<i class="fas fa-chevron-right"></i>',
			'type' => 'list',
			'add_args' => false,
			'add_fragment' => ''
		);

		$paginate_links = paginate_links($pagination_args);
		if ($paginate_links) {
			echo '<div class="col-lg-12 col-md-12 col-sm-12">
					<div class="pagination justify-content-center">
								<nav aria-label="Page navigation">';
			printf(esc_html__('%s', 'power-hour'), $paginate_links);
			echo '</nav>
					</div>
				</div>';
		}
	}

	public function powerhour_inner_breadcrumb()
	{
		$powerhour_option = get_option('power-hour-options');
		$page_id = get_queried_object_id();
		$key = (function_exists('get_field')) ? get_field('key_pjros12gdfgd', $page_id) : "";

		if (get_post_type() == "portfolio" && $key['iqonic_breadcrumb'] == 'no') { 
			return;
		} 

		if(class_exists('WooCommerce') && is_product() && $powerhour_option['product_display_banner'] == "no") {
			return;
		}
		
		if (!is_front_page() && !is_404()) {

			if (is_page() && has_post_thumbnail(get_queried_object_id())) {
				$options = '';

				if (isset($powerhour_option['bg_opacity'])) {
					$options = $powerhour_option['bg_opacity'];
				}

				if ($options == "1") {
					$bg_class = 'powerhour-bg-over black';
				} elseif ($options == "2") {
					$bg_class = 'powerhour-bg-over powerhour-over-dark-80';
				} elseif ($options == "3") {
					$bg_class = 'breadcrumb-bg breadcrumb-ui';
				} else {
					$bg_class = 'powerhour-bg-over powerhour-over-dark-80';
				}
                  ?>
				<div class="text-left powerhour-breadcrumb-one <?php if (!empty($bg_class)) {
																	echo esc_attr($bg_class);
																} ?>">
				<?php

			} else {

				if (!empty($powerhour_option['bg_type']) && $powerhour_option['bg_type'] == "1") {

					$bg_color = esc_html__('powerhour-bg-over black', 'power-hour');
				} elseif (!empty($powerhour_option['bg_type']) && $powerhour_option['bg_type'] == "2") {

					if (isset($powerhour_option['banner_image']['url'])) {
						$bgurl = $powerhour_option['banner_image']['url'];
					}
					$options = $powerhour_option['bg_opacity'];

					if ($options == "1") {
					} elseif ($options == "2") {
						$bg_class = 'powerhour-bg-over powerhour-over-dark-80';
					} elseif ($options == "3") {
						$bg_class = 'breadcrumb-bg breadcrumb-ui';
					} else {
						$bg_class = 'powerhour-bg-over powerhour-over-dark-80';
					}
				} else {
					$bg_class = 'powerhour-bg-over';
				} ?>

					<div class="powerhour-breadcrumb-one <?php if (isset($powerhour_option['bg_type']) && !empty($powerhour_option['bg_type'] == "1")) {
																echo esc_attr($bg_color);
															} ?> <?php if (!empty($bg_class)) {
																		echo esc_attr($bg_class);
																	} ?>"> <?php
																		} ?>
						<div class="container">
							<?php

							$options = '';

							if (!empty($powerhour_option['bg_image'])) {
								$options = $powerhour_option['bg_image'];
							}

							if ($options == '1' && class_exists('ReduxFramework')) {    ?>
								<div class="row align-items-center justify-content-center text-center">
									<div class="col-sm-12">
										<nav aria-label="breadcrumb" class=" powerhour-breadcrumb-two">
											<?php
											$this->powerhour_breadcrumbs_title();
											if (isset($powerhour_option['display_breadcrumbs'])) {
												$options = $powerhour_option['display_breadcrumbs'];
												if ($options == "yes") {
											?>
													<ol class="breadcrumb main-bg">
														<?php $this->powerhour_custom_breadcrumbs(); ?>
													</ol>
											<?php
												}
											}

											?>
										</nav>
									</div>
								</div>
								<?php

							} elseif ($options == '2' && class_exists('ReduxFramework')) { ?>

								<div class="row align-items-center">
									<div class="col-lg-8 col-md-8 text-left align-self-center">
										<nav aria-label="breadcrumb" class="text-left">
											<?php
											$this->powerhour_breadcrumbs_title();
											if (isset($powerhour_option['display_breadcrumbs'])) {
												$options = $powerhour_option['display_breadcrumbs'];
												if ($options == "yes") { ?>
													<ol class="breadcrumb main-bg">
														<?php $this->powerhour_custom_breadcrumbs(); ?>
													</ol> <?php
														}
													}
															?>
										</nav>
									</div>
									<div class="col-lg-4 col-md-4 col-sm-12 text-right wow fadeInRight">
										<?php $this->powerhour_breadcrumbs_feature_image(); ?>
									</div>
								</div>
								<?php

							} elseif ($options == '3' && class_exists('ReduxFramework')) { ?>

								<div class="row align-items-center">
									<div class="col-lg-4 col-md-4 col-sm-12 wow fadeInLeft">
										<?php $this->powerhour_breadcrumbs_feature_image(); ?>
									</div>
									<div class="col-lg-8 col-md-8 text-left align-self-center">
										<nav aria-label="breadcrumb" class="text-right powerhour-breadcrumb-two">
											<?php
											$this->powerhour_breadcrumbs_title();

											if (isset($powerhour_option['display_breadcrumbs'])) {
												$options = $powerhour_option['display_breadcrumbs'];
												if ($options == "yes") { ?>
													<ol class="breadcrumb main-bg">
														<?php $this->powerhour_custom_breadcrumbs(); ?>
													</ol> <?php
														}
													}
															?>
										</nav>
									</div>
								</div>
							<?php

							} elseif ($options == '4' && class_exists('ReduxFramework')) { ?>

								<div class="row align-items-center powerhour-breadcrumb-three">
									<div class="col-sm-6 mb-3 mb-lg-0 mb-md-0">
										<?php $this->powerhour_breadcrumbs_title(); ?>
									</div>
									<div class="col-sm-6 ext-lg-right text-md-right text-sm-left">
										<nav aria-label="breadcrumb" class="powerhour-breadcrumb-two">
											<?php
											if (isset($powerhour_option['display_breadcrumbs'])) {
												$options = $powerhour_option['display_breadcrumbs'];
												if ($options == "yes") { ?>
													<ol class="breadcrumb main-bg">
														<?php $this->powerhour_custom_breadcrumbs(); ?>
													</ol> <?php
														}
													} ?>
										</nav>
									</div>
								</div>
								<?php

							} elseif ($options == '5' && class_exists('ReduxFramework')) { ?>

								<div class="row align-items-center powerhour-breadcrumb-three">
									<div class="col-sm-6 mb-3 mb-lg-0 mb-md-0">
										<nav aria-label="breadcrumb" class="text-left powerhour-breadcrumb-two">
											<div class="col-sm-6 text-right">
												<?php $this->powerhour_breadcrumbs_title(); ?>
											</div>
											<?php
											if (isset($powerhour_option['display_breadcrumbs'])) {
												$options = $powerhour_option['display_breadcrumbs'];
												if ($options == "yes") {
											?>
													<ol class="breadcrumb main-bg">
														<?php $this->powerhour_custom_breadcrumbs(); ?>
													</ol>
											<?php
												}
											}
											?>
										</nav>
									</div>

								</div>
								<?php

							} else { ?>
								<div class="row align-items-center">
									<div class="col-sm-12">
										<nav aria-label="breadcrumb" class="text-center">
											<?php $this->powerhour_breadcrumbs_title(); ?>
											<ol class="breadcrumb main-bg">
												<?php $this->powerhour_custom_breadcrumbs(); ?>
											</ol>
										</nav>
									</div>
								</div> <?php

							} ?>

						</div>
					</div>
				<?php
			}
		}

		function powerhour_breadcrumbs_title()
		{
			$powerhour_options = get_option('power-hour-options');
			$title_tag = 'h3';
			$title = '';
			$id = (get_queried_object_id()) ? get_queried_object_id() : '';

			if (function_exists('get_field')) {
				$id = get_field('acf_blog_detail_style', $id);
			}

			if (isset($powerhour_options['breadcum_title_tag'])) {
				$title_tag = $powerhour_options['breadcum_title_tag'];
			}

			if ($id == 3) { ?>
				<div class="powerhour-breadcrumb-bd single_style-3">
					<a href="<?php echo get_day_link(get_post_time('Y'), get_post_time('m'), get_post_time('j')); ?>" class="entry-date">
						<?php the_time('F j, Y') ?>
					</a>
				</div> <?php
			} elseif (isset($powerhour_options['blog_detail_style']) && $powerhour_options['blog_detail_style'] ==  3) { ?>
				<div class="powerhour-breadcrumb-bd single_style-3">
					<a href="<?php echo get_day_link(get_post_time('Y'), get_post_time('m'), get_post_time('j')); ?>" class="entry-date">
						<?php the_time('F j, Y') ?>
					</a>
				</div>
			<?php }

			if (is_archive()) {
				if( get_post_type() == 'service' ){
					$title = esc_html__('Service', 'power-hour');
				} else {
					$title = get_the_archive_title();
				}
			} elseif (is_search()) {
				if (have_posts()) :
					$title = esc_html__('Search Results for:', 'power-hour');
					$title = $title . ' <span>' . get_search_query() . '</span>';
				else :  $title = esc_html__('Nothing Found', 'power-hour');
				endif;
			} elseif (is_404()) {
				if (isset($powerhour_options['power-hour_fourzerofour_title'])) {
					$title = $powerhour_options['power-hour_fourzerofour_title'];
				} else {
					$title = esc_html__('Oops! That page can not be found.', 'power-hour');
				}
			} elseif (is_home()) {
				$title = wp_title('', false);
			} elseif ('iqonic_hf_layout' === get_post_type()) {
				$title = get_the_title($id);
			} else {
				$id = (get_queried_object_id()) ? get_queried_object_id() : '';
				if (get_post_type($id) == 'team') {
					$title = get_the_title($id);
				} elseif (get_post_type($id) == 'service') {
					$title = get_the_title($id);
				} elseif (get_post_type($id) == 'portfolio') {
					$title = "Portfolio Details";
				} elseif (get_post_type($id) == 'tribe_events') {
					$title = "Event Details";
				} elseif ('post' === get_post_type()) {
					if ($id == 3) {
						$title = get_the_title();
					} elseif (isset($powerhour_options['blog_detail_style']) && $powerhour_options['blog_detail_style'] == 3) {
						$title = get_the_title();
					} else {
						$title = __('Blog post', 'power-hour');
					}
				} else {
					$title = get_the_title();
				}
			}
				?>
				<<?php echo esc_attr($title_tag); ?> class="title">
					<?php echo wp_kses_post($title); ?>
				</<?php echo esc_attr($title_tag); ?>>
				<?php
			}

			function powerhour_breadcrumbs_feature_image()
			{
				$powerhour_options = get_option('power-hour-options');
				$bnurl = '';
				$page_id = get_queried_object_id();
				if (has_post_thumbnail($page_id) && !is_single()) {
					$image_array = wp_get_attachment_image_src(get_post_thumbnail_id($page_id), 'full');
					$bnurl = $image_array[0];
				} elseif (is_404()) {
					if (!empty($powerhour_options['404_banner_image']['url'])) {
						$bnurl = $powerhour_options['404_banner_image']['url'];
					}
				} elseif (is_home()) {
					if (!empty($powerhour_options['blog_default_banner_image']['url'])) {
						$bnurl = $powerhour_options['blog_default_banner_image']['url'];
					}
				} else {
					if (!empty($powerhour_options['page_default_banner_image']['url'])) {
						$bnurl = $powerhour_options['page_default_banner_image']['url'];
					}
				}

				if (!empty($bnurl)) {
				?>
					<img src="<?php echo esc_url($bnurl); ?>" 
						class="img-fluid <?php if (!empty($powerhour_options['bg_image'])) {
							if (!$powerhour_options['bg_image'] == 1) {
								echo 'float-right';
							}
						} ?>" alt="<?php esc_attr_e('banner', 'power-hour'); ?>">
		            <?php
				}
			}
			function powerhour_custom_breadcrumbs()
			{

				$show_on_home = 0; // 1 - show breadcrumbs on the homepage, 0 - don't show
				$home = '' . esc_html__('Home', 'power-hour') . ''; // text for the 'Home' link
				$show_current = 1; // 1 - show current post/page title in breadcrumbs, 0 - don't show

				global $post;
				$home_link = esc_url(home_url());
				$svg_icon = '<div class="circle-box">
					<div class="sub-circle">
						<svg width="17" height="11" viewBox="0 0 17 11" fill="none" xmlns="http://www.w3.org/2000/svg">
						<path d="M13.1021 7.43909V9.6188H12.0239V7.43909H13.1021Z" fill="#222222"/>
						<path d="M13.1021 1.172V3.35171L12.0239 3.35171V1.172L13.1021 1.172Z" fill="#222222"/>
						<path d="M16.8755 4.98694V6.07679L14.7192 6.07679V4.98694L16.8755 4.98694Z" fill="#222222"/>
						<path d="M9.28882 4.57812V6.21291L0.124692 6.21291V4.57813L9.28882 4.57812Z" fill="#222222"/>
						<path d="M4.92438 8.92429L8.1825 5.39534L4.92546 1.86639L6.10709 0.751465L9.88056 4.83842C10.1717 5.1523 10.1717 5.63946 9.88056 5.95334L6.10709 10.0403L4.92438 8.92429Z" fill="#222222"/>
						</svg>
					</div>
                </div>';

				if (is_front_page()) {
					if ($show_on_home == 1) echo '<li class="breadcrumb-item"><a href="' . $home_link . '">' . $home . '</a></li>';
				} else {

					echo '<li class="breadcrumb-item"><a href="' . $home_link . '">' . $home . '</a></li> ';

					if (is_home()) {
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Blogs', 'power-hour') . '</li>';
					} elseif (is_category()) {
						$this_cat = get_category(get_query_var('cat'), false);
						if ($this_cat->parent != 0) echo '<li class="breadcrumb-item">' . $svg_icon . '' . get_category_parents($this_cat->parent, TRUE, '  ') . '</li>';
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Archive by category : ', 'power-hour') . ' "' . single_cat_title('', false) . '" </li>';
					} elseif (is_search()) {
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Search results for : ', 'power-hour') . ' "' . get_search_query() . '"</li>';
					} elseif (is_day()) {
						echo '<li class="breadcrumb-item active">' . $svg_icon . '' . get_the_date('F d, Y') . '</li>';
					} elseif (is_month()) {
						echo '<li class="breadcrumb-item active">' . $svg_icon . '' . get_the_date('F Y') . '</li>';
					} elseif (is_year()) {
						echo '<li class="breadcrumb-item active">' . $svg_icon . '' . get_the_date('Y') . '</li>';
					}
					
					elseif (is_single() && !is_attachment()) {
						$id = (get_queried_object_id()) ? get_queried_object_id() : '';
						if (get_post_type($id) != 'post') {
							$post_type = get_post_type_object(get_post_type($id));
							$slug = $post_type->rewrite;
							if (!empty($slug)) {
								echo '<li class="breadcrumb-item">' . $svg_icon . '<a href="' . $home_link . '/' . $slug['slug'] . '/">' . $post_type->labels->singular_name . '</a></li>';
							}
							if ($show_current == 1) echo '<li class="breadcrumb-item test">' . $svg_icon . '' . get_the_title($id) . '</li>';
						} else {
							$cat = get_the_category($id);
							if (!empty($cat)) {
								$cat = $cat[0];
								if ($show_current == 0) $cat = preg_replace("#^(.+)\s\s$#", "$1", $cat);
								echo '<li class="breadcrumb-item">' . $svg_icon . '' . get_category_parents($cat, TRUE, '  ') . '</li>';
								if (!empty(get_the_title($id))) {
									if ($show_current == 1) echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . get_the_title($id) . '</li>';
								}
							}
						}
					} elseif (!is_single() && !is_page() && get_post_type() != 'post' && !is_404()) {
						$post_type = get_post_type_object(get_post_type());
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . $post_type->labels->singular_name . '</li>';
					} elseif (!is_single() && is_attachment()) {
						$id = (get_queried_object_id()) ? get_queried_object_id() : '';
						$parent = get_post($post->post_parent);
						$cat = get_the_category($parent->ID);
						$cat = $cat[0];
						echo '<li class="breadcrumb-item">' . $svg_icon . '' . get_category_parents($cat, TRUE, '  ') . '</li>';
						echo '<li class="breadcrumb-item">' . $svg_icon . '' . '<a href="' . get_permalink($parent) . '">' . $parent->post_title . '</a></li>';
						if ($show_current == 1) echo '<li class="breadcrumb-item active"> ' . $svg_icon . '' .  get_the_title($id) . '</li>';
					} elseif (is_page() && !$post->post_parent) {
						$id = (get_queried_object_id()) ? get_queried_object_id() : '';
						if ($show_current == 1) echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . get_the_title($id) . '</li>';
					} elseif (is_page() && $post->post_parent) {
						$trail = '';
						$id = (get_queried_object_id()) ? get_queried_object_id() : '';
						if ($post->post_parent) {
							$parent_id = $post->post_parent;
							$breadcrumbs = array();
							while ($parent_id) {
								$page = get_post($parent_id);
								$breadcrumbs[] = '<li class="breadcrumb-item">' . $svg_icon . '' . '<a href="' . get_permalink($page->ID) . '">' . get_the_title($page->ID) . '</a></li>';
								$parent_id  = $page->post_parent;
							}
							$breadcrumbs = array_reverse($breadcrumbs);
							foreach ($breadcrumbs as $crumb) $trail .= $crumb;
						}

						printf($trail);
						if ($show_current == 1) echo '<li class="breadcrumb-item active"> ' . $svg_icon . '' .  get_the_title($id) . '</li>';
					} elseif (is_tag()) {
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Posts tagged', 'power-hour') . ' "' . single_tag_title('', false) . '"</li>';
					} elseif (is_author()) {
						global $author;
						$userdata = get_userdata($author);
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Articles posted by : ', 'power-hour') . ' ' . $userdata->display_name . '</li>';
					} elseif (is_404()) {
						echo  '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Error 404', 'power-hour') . '</li>';
					}

					if (get_query_var('paged')) {
						echo '<li class="breadcrumb-item active">' . $svg_icon . '' . esc_html__('Page', 'power-hour') . ' ' . get_query_var('paged') . '</li>';
					}
				}
			}

			function powerhour_searchfilter($query)
			{
				if (!is_admin()) {
					if ($query->is_search) {
						$query->set('post_type', 'post');
					}
					return $query;
				}
			}


	/**
	 * Adds a container opening div with a dynamic class for specific conditions.
	 *
	 * @param string $output The default output to be modified.
	 * @return string The modified output with the container div or empty string.
	 */
	function powerhour_add_container_open_if_redux($output) {

		if (! class_exists('Iqonic\Classes\Iqonic_Extension')) {
			return '<div class="' . apply_filters('content_container_class', 'container') . '">';
		}
	
		if (class_exists('Iqonic\Classes\Iqonic_Extension')) {
	
			// Allowed post types
			$allowed_post_types = [
				'post',
				'portfolio',
				'Service',
				'team',
				'testimonial'
			];
	
			// Get current post type
			$post_type = get_post_type();
			if (is_post_type_archive() || is_tax()) {
				$post_type = get_query_var('post_type');
			}
	
			// Check post type and return container opening
			if (in_array($post_type, $allowed_post_types, true) || is_search()) {
				return '<div class="' . apply_filters('content_container_class', 'container') . '">';
			}
	
			global $post;
	
			// Get the Wishlist page ID from options.
			$wishlist_page_id = get_option('yith_wcwl_wishlist_page_id');
	
			// List of conditions to apply the 'container' class.
			if (
				(function_exists('is_account_page') && is_account_page()) || // WooCommerce My Account page
				(function_exists('is_cart') && is_cart()) ||   // WooCommerce Cart page
				(function_exists('is_checkout') && is_checkout()) ||        // WooCommerce Checkout page
				(is_page() && isset($post->post_name) && $post->post_name === 'order-tracking') || // Order Tracking page
				(get_the_ID() === intval($wishlist_page_id)) // Wishlist page
			) {
				return '<div class="' . apply_filters('content_container_class', 'container') . '">';
			}
		}
	
		return '';
	}

	/**
	 * Adds a container closing div for specific conditions.
	 *
	 * @param string $output The default output to be modified.
	 * @return string The modified output with the closing div or empty string.
	 */
	function powerhour_add_container_close_if_redux($output) {
	
		if (! class_exists('Iqonic\Classes\Iqonic_Extension')) {
			return '</div>';
		}
	
		if (class_exists('Iqonic\Classes\Iqonic_Extension')) {
	
			$allowed_post_types = [
				'post',
				'portfolio',
				'Service',
				'team',
				'testimonial'
			];
	
			$post_type = get_post_type();
			if (is_post_type_archive() || is_tax()) {
				$post_type = get_query_var('post_type');
			}
	
			if (in_array($post_type, $allowed_post_types, true) || is_search()) {
				return '</div>';
			}
	
			global $post;
	
			// Get the Wishlist page ID from options.
			$wishlist_page_id = get_option('yith_wcwl_wishlist_page_id');
	
			// List of conditions to apply the 'container' class.
			if (
				(function_exists('is_account_page') && is_account_page()) || // WooCommerce My Account page
				(function_exists('is_cart') && is_cart()) ||   // WooCommerce Cart page
				(function_exists('is_checkout') && is_checkout()) ||        // WooCommerce Checkout page
				(is_page() && isset($post->post_name) && $post->post_name === 'order-tracking') || // Order Tracking page
				(get_the_ID() === intval($wishlist_page_id)) // Wishlist page
			) {
				return '</div>';
			}
		}
		return '';
	}

}
