<?php

/*
Plugin Name: WPBooked
Plugin URI: https://wpbooked.iqonic.design/
Description: Take control of your calendar with our efficient booking solution.
Version: 1.1.0
Author: Iqonic Design
Author URI: https://iqonic.design/
Text Domain: wpbooked
*/

define( 'WPBOOKED_VERSION', '1.1.0' );
define( 'WPBOOKED_WELCOME_SCREEN', get_option('wpbooked_welcome_screen',true) );
define( 'WPBOOKED_DEMO_MODE', get_option('wpbooked_demo_mode',false) );
define( 'WPBOOKED_PLUGIN_URL', untrailingslashit( plugin_dir_url( __FILE__ ) ) );
define( 'WPBOOKED_PLUGIN_DIR', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define( 'WPBOOKED_STYLESHEET_DIR', get_stylesheet_directory() );
define( 'WPBOOKED_PLUGIN_TEMPLATES_DIR', WPBOOKED_PLUGIN_DIR . '/templates/' );
define( 'WPBOOKED_AJAX_INCLUDES_DIR', WPBOOKED_PLUGIN_DIR . '/includes/ajax/' );

// FontAwesome Support
require_once __DIR__ . '/vendor/fortawesome/wordpress-fontawesome/index.php';

// Included Add-Ons
require_once WPBOOKED_PLUGIN_DIR . '/includes/add-ons/init.php';

// Booked Mailer Functions
require_once('includes/mailer_functions.php');

if(!class_exists('wpbooked_plugin')) {
	class wpbooked_plugin {

		public $booked_screens;

		/**
		 * Construct the plugin object
		 */
		public function __construct() {

			$this->booked_screens = apply_filters('wpbooked_admin_booked_screens', array('wpbooked-pending','wpbooked-appointments','wpbooked-settings','wpbooked-welcome'));

			require_once(sprintf("%s/post-types/booked_appointments.php", WPBOOKED_PLUGIN_DIR));
			$booked_appointments_post_type = new wpbooked_appointments_post_type();

			require_once(sprintf("%s/includes/general-functions.php", WPBOOKED_PLUGIN_DIR));
			require_once(sprintf("%s/includes/shortcodes.php", WPBOOKED_PLUGIN_DIR));
			require_once(sprintf("%s/includes/widgets.php", WPBOOKED_PLUGIN_DIR));
			require_once(sprintf("%s/includes/ajax/init.php", WPBOOKED_PLUGIN_DIR));
			require_once(sprintf("%s/includes/ajax/init_admin.php", WPBOOKED_PLUGIN_DIR));
			//require migration class file
			require_once(sprintf("%s/includes/migration.php", WPBOOKED_PLUGIN_DIR));

			$booked_ajax = new WPBooked_AJAX();
			$booked_admin_ajax = new WPBooked_Admin_AJAX();

			add_action('admin_init', array($this, 'admin_init'), 9);
			add_action('admin_menu', array($this, 'add_menu'));
			add_action('admin_bar_menu', array($this, 'add_admin_bar_menu'), 65);

			add_action('the_posts', array($this, 'add_to_calendar_check'));

			add_action('admin_enqueue_scripts', array($this, 'admin_styles'));
			add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
			add_action('manage_users_custom_column', array($this, 'wpbooked_add_custom_user_columns'), 15, 3);
			add_filter('manage_users_columns', array($this, 'wpbooked_add_user_columns'), 15, 1);
			add_filter('user_contactmethods', array($this, 'wpbooked_phone_numbers'));

			add_action('wpbooked_profile_tabs', array($this, 'wpbooked_profile_tabs'));
			add_action('wpbooked_profile_tab_content', array($this, 'wpbooked_profile_tab_content'));
			add_action('wp_enqueue_scripts', array($this, 'front_end_scripts'),1);

			add_action('admin_menu', array($this, 'wpbooked_add_pending_appt_bubble' ));
			add_action('admin_notices', array($this, 'wpbooked_pending_notice' ));
			add_action('admin_notices', array($this, 'wpbooked_no_profile_page_notice' ));
			add_action('parent_file', array($this, 'wpbooked_tax_menu_correction'));

			add_action( 'wpbooked_custom_calendars_add_form_fields', array($this, 'wpbooked_calendars_add_custom_fields'), 10, 2 );
			add_action( 'wpbooked_custom_calendars_edit_form_fields', array($this, 'wpbooked_calendars_edit_custom_fields'), 10, 2 );
			add_action( 'create_booked_custom_calendars', array($this, 'wpbooked_save_calendars_custom_fields'), 10, 2 );
			add_action( 'edited_booked_custom_calendars', array($this, 'wpbooked_save_calendars_custom_fields'), 10, 2 );

			add_action('init', array($this, 'init'),10);

			// Prevent WooCommerce from Redirecting "Booking Agents" to the My Account page.
			add_filter('woocommerce_prevent_admin_access', array($this, 'wpbooked_wc_check_admin_access'));

			// Allow other plugins/themes to apply Booked capabilities to other user roles
			add_filter( 'wpbooked_user_roles', array($this,'wpbooked_user_roles_filter') );

			// Email Reminders (Added in v1.8.0)
			add_filter( 'cron_schedules', array($this,'cron_schedules'));
			add_action( 'wpbooked_send_admin_reminders', array($this, 'admin_reminders'), 20 );
			add_action( 'wpbooked_send_user_reminders', array($this, 'user_reminders'), 20 );

			// Add new column
			add_filter('manage_booked_appointments_posts_columns', array($this,'wpbook_add_new_column'));
			add_action('manage_booked_appointments_posts_custom_column', array($this,'wpbook_add_new_column_data'), 10, 2);

			$user_email_content = get_option('wpbooked_reminder_email',false);
			$user_email_subject = get_option('wpbooked_reminder_email_subject',false);

			if ($user_email_content && $user_email_subject):
				if ( !wp_next_scheduled('wpbooked_send_user_reminders') ):
					wp_schedule_event( time(),'wpbooked_everyfive','wpbooked_send_user_reminders' );
			    endif;
			else:
				wp_clear_scheduled_hook( 'wpbooked_send_user_reminders' );
			endif;

			$admin_email_content = get_option('wpbooked_admin_reminder_email',false);
			$admin_email_subject = get_option('wpbooked_admin_reminder_email_subject',false);

			if ($admin_email_content && $admin_email_subject):
				if ( !wp_next_scheduled('wpbooked_send_admin_reminders') ):
					wp_schedule_event(time(),'wpbooked_everyfive','wpbooked_send_admin_reminders');
			    endif;
			else:
				wp_clear_scheduled_hook('wpbooked_send_admin_reminders');
			endif;

		}

		public function wpbook_add_new_column($columns) {
			// Replace 'book' with your custom post type slug
			$columns['action'] = 'Action';
			return $columns;
		}

		public function wpbook_add_new_column_data($column, $post_id) {
			// Replace 'book' with your custom post type slug and 'action' with your column name
			if ($column == 'action') {
				$post_status = get_post_status($post_id);
				
				if($post_status != "publish"){
					// Retrieve data for your Action
					echo '<div class="pending-appt WPbookedClearFix" data-appt-id="'.esc_html($post_id).'">';
					echo '<button data-appt-id="'.esc_html($post_id).'" class="approve button button-primary">'.esc_html__('Approve','wpbooked').'</button>';
					echo '<button style="margin-left:10px" class="delete button">'.esc_html__('Delete','wpbooked').'</button>';
					echo '</div>';
				
				}else{
					echo "This appointment Approved";
				}

				
			}
		}

		public static function admin_reminders(){

			$admin_reminder_buffer = get_option('wpbooked_admin_reminder_buffer',30);
			$start_timestamp = current_time('timestamp');
			$end_timestamp = strtotime(date_i18n('Y-m-d H:i:s',current_time('timestamp')).' + '.$admin_reminder_buffer.' minutes');

			$args = array(
				'post_type' => 'wpbooked_apts',
				'posts_per_page' => 500,
				'post_status' => array('publish','future'),
				'meta_query' => array(
					array(
						'key'     => '_appointment_timestamp',
						'value'   => array( $start_timestamp, $end_timestamp ),
						'compare' => 'BETWEEN',
					)
				)
			);

			$bookedAppointments = new WP_Query($args);

			if( $bookedAppointments->have_posts() ):
				while ( $bookedAppointments->have_posts() ):

					$bookedAppointments->the_post();
					global $post;

					$appt_id = $post->ID;
					$reminder_sent = get_post_meta($appt_id,'_appointment_admin_reminder_sent',true);

					$calendars = get_the_terms( $appt_id, 'wpbooked_custom_calendars' );
					if ( !empty($calendars) ):
						foreach( $calendars as $calendar ):
							$calendar_id = $calendar->term_id;
						endforeach;
					else:
						$calendar_id = false;
					endif;

					if ( !$reminder_sent && apply_filters( 'wpbooked_prepare_sending_reminder', true, $appt_id ) ):

						$email_content = get_option('wpbooked_admin_reminder_email',false);
						$email_subject = get_option('wpbooked_admin_reminder_email_subject',false);
						if ($email_content && $email_subject):

							$admin_email = wpbooked_which_admin_to_send_email( $calendar_id );
							$token_replacements = wpbooked_get_appointment_tokens( $appt_id );
							$email_content = wpbooked_token_replacement( $email_content,$token_replacements );
							$email_subject = wpbooked_token_replacement( $email_subject,$token_replacements );

							update_post_meta($appt_id,'_appointment_admin_reminder_sent',true);

							do_action( 'wpbooked_admin_reminder_email', $admin_email, $email_subject, $email_content, $token_replacements['email'], $token_replacements['name'] );

						endif;

					endif;

				endwhile;

			endif;

			wp_reset_postdata();

		}

		public static function user_reminders(){

			$user_reminder_buffer = get_option('wpbooked_reminder_buffer',30);

			$start_timestamp = current_time('timestamp');
			$end_timestamp = strtotime(date_i18n('Y-m-d H:i:s',current_time('timestamp')).' + '.$user_reminder_buffer.' minutes');

			$args = array(
				'post_type' => 'wpbooked_apts',
				'posts_per_page' => 500,
				'post_status' => array('publish','future'),
				'meta_query' => array(
					array(
						'key'     => '_appointment_timestamp',
						'value'   => array( $start_timestamp, $end_timestamp ),
						'compare' => 'BETWEEN',
					)
				)
			);

			$bookedAppointments = new WP_Query($args);
			if($bookedAppointments->have_posts()):
				while ($bookedAppointments->have_posts()):

					$bookedAppointments->the_post();
					global $post;

					$appt_id = $post->ID;
					$reminder_sent = get_post_meta($appt_id,'_appointment_user_reminder_sent',true);

					$send_mail = true;
					if ( !$reminder_sent && apply_filters( 'wpbooked_prepare_sending_reminder', true, $appt_id ) ):

						$email_content = get_option('wpbooked_reminder_email',false);
						$email_subject = get_option('wpbooked_reminder_email_subject',false);

						if ($email_content && $email_subject):

							$token_replacements = wpbooked_get_appointment_tokens( $appt_id );
							$email_content = wpbooked_token_replacement( $email_content,$token_replacements );
							$email_subject = wpbooked_token_replacement( $email_subject,$token_replacements );

							update_post_meta($appt_id,'_appointment_user_reminder_sent',true);

							do_action( 'wpbooked_reminder_email', $token_replacements['email'], $email_subject, $email_content );

						endif;

					endif;

				endwhile;

			endif;

			wp_reset_postdata();

		}

		public static function cron_schedules( $schedules ) {
			$schedules['wpbooked_everyfive'] = array(
				'interval' => 60 * 5,
				'display' => esc_html__('Every Five Minutes', 'wpbooked')
			);

			return $schedules;
		}

		public static function activate() {
			set_transient( '_booked_welcome_screen_activation_redirect', true, 30 );
		}

		public function wpbooked_wc_check_admin_access( $redirect_to ) {
			$booked_current_user = wp_get_current_user();
			if( is_array( $booked_current_user->roles ) && in_array( 'wpbooked_booking_agent', $booked_current_user->roles ) ) {
				return false;
  			}
  			return $redirect_to;
		}

		public function admin_init() {

			if (current_user_can('edit_booked_appointments') && isset($_POST['wpbooked_export_appointments_csv'])):
				include('includes/export-csv.php');
			endif;

			$booked_redirect_non_admins = get_option('wpbooked_redirect_non_admins',false);

			// Redirect non-admin users
			if ($booked_redirect_non_admins):
				if (!current_user_can('edit_booked_appointments') && !defined( 'DOING_AJAX' )){

					$booked_profile_page = wpbooked_get_profile_page();

					if ($booked_profile_page):
						$redirect_url = get_permalink($booked_profile_page);
					else:
						$redirect_url = home_url();
					endif;

					wp_redirect( $redirect_url );
					exit;

				}
			endif;

			// Set up the settings for this plugin
			require_once(sprintf("%s/includes/admin-functions.php", WPBOOKED_PLUGIN_DIR));
			require_once(sprintf("%s/includes/dashboard-widget.php", WPBOOKED_PLUGIN_DIR));
			$this->init_settings();

			// Welcome Screen Redirect
			if ( !get_transient( '_booked_welcome_screen_activation_redirect' ) ) {
				return;
  			}

  			delete_transient( '_booked_welcome_screen_activation_redirect' );

  			if ( is_network_admin() || isset( $_GET['activate-multi'] ) ) {
  				return;
  			}

  			return;

		} // END public static function activate

		public function wpbooked_curl($url){

			if ( function_exists('curl_init') ):

				$ch = curl_init();
				$timeout = 5;
				curl_setopt($ch, CURLOPT_URL, $url);
				curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
				curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
				$data = curl_exec($ch);
				curl_close($ch);
				if ($data):
					return $data;
				else:
					return false;
				endif;

			else:

				return false;

			endif;

		}

		public function wpbooked_profile_tabs($default_tabs){

			foreach($default_tabs as $slug => $name):
				echo '<li'.esc_html($name['class'] ? ' class="'.$name['class'].'"' : '').'><a href="#'.esc_html($slug).'"><i class="fa-solid fa-'.esc_html($name['fa-icon']).'"></i>'.wp_kses_post($name['title']).'</a></li>';
			endforeach;

		}

		public function wpbooked_profile_tab_content($default_tabs){

			foreach($default_tabs as $slug => $name):
				echo '<div id="profile-'.esc_html($slug).'" class="wpbooked-tab-content WPbookedClearFix">';
					call_user_func('wpbooked_profile_content_'.esc_html($slug));
				echo '</div>';
			endforeach;

		}

		public function init() {

			// Hide the Admin Bar from subscribers.
		    $booked_current_user = wp_get_current_user();
			if ( isset($booked_current_user->roles[0]) && in_array( 'subscriber',$booked_current_user->roles ) ) {
				add_filter('show_admin_bar', '__return_false');
			}

			// Include the Booked functions file.
			require_once(sprintf("%s/includes/functions.php", WPBOOKED_PLUGIN_DIR));

			// Start a session if none is started yet.
			if( !headers_sent() && !(session_id()  && apply_filters( 'wpbooked_sessions_enabled', true )) ) {
		        session_start();
		    }

		    // Check to see if the plugin was updated.
			$current_version = get_option('wpbooked_version_check','1.6.20');
			if ( version_compare( $current_version, WPBOOKED_VERSION ) < 0 && !WPBOOKED_DEMO_MODE):
				update_option('wpbooked_version_check',WPBOOKED_VERSION);
				set_transient( '_booked_welcome_screen_activation_redirect', true, 60 );
				set_transient('wpbooked_show_new_tags',true,60*60*24*15);
			else:
				update_option('wpbooked_version_check',WPBOOKED_VERSION);
			endif;

			//Site health issues fix - patch
			session_write_close();
		}

		public function add_to_calendar_check($posts){

		    if (empty($posts)):
		    	return $posts;
		    endif;

		    $found = false;

			foreach ($posts as $post):

				$profile_shortcode = stripos($post->post_content, '[wpbooked-profile');
				$appts_shortcode = stripos($post->post_content, '[wpbooked-appointments');

				if ( $profile_shortcode !== false || $appts_shortcode !== false):
					$found = true;
					break;
				endif;

			endforeach;

		    return $posts;

		}

		static function plugin_settings() {
			$plugin_options = array(
				'wpbooked_login_redirect_page',
				'wpbooked_custom_login_message',
				'wpbooked_appointment_redirect_type',
				'wpbooked_appointment_success_redirect_page',
				'wpbooked_registration_name_requirements',
				'wpbooked_hide_admin_bar_menu',
				'wpbooked_timeslot_intervals',
				'wpbooked_appointment_buffer',
				'wpbooked_appointment_limit',
				'wpbooked_cancellation_buffer',
				'wpbooked_new_appointment_default',
				'wpbooked_prevent_appointments_before',
				'wpbooked_prevent_appointments_after',
				'wpbooked_booking_type',
				'wpbooked_require_guest_email_address',
				'wpbooked_hide_default_calendar',
				'wpbooked_hide_unavailable_timeslots',
				'wpbooked_hide_google_link',
				'wpbooked_hide_weekends',
				'wpbooked_dont_allow_user_cancellations',
				'wpbooked_show_only_titles',
				'wpbooked_hide_end_times',
				'wpbooked_hide_available_timeslots',
				'wpbooked_public_appointments',
				'wpbooked_redirect_non_admins',
				'wpbooked_light_color',
				'wpbooked_dark_color',
				'wpbooked_button_color',
				'wpbooked_email_logo',
				'wpbooked_default_email_user',
				'wpbooked_email_force_sender',
				'wpbooked_email_force_sender_from',
				'wpbooked_emailer_disabled',
				'wpbooked_reminder_buffer',
				'wpbooked_admin_reminder_buffer',
				'wpbooked_reminder_email',
				'wpbooked_admin_reminder_email',
				'wpbooked_reminder_email_subject',
				'wpbooked_admin_reminder_email_subject',
				'wpbooked_registration_email_subject',
				'wpbooked_registration_email_content',
				'wpbooked_approval_email_content',
				'wpbooked_approval_email_subject',
				'wpbooked_cancellation_email_content',
				'wpbooked_cancellation_email_subject',
				'wpbooked_appt_confirmation_email_content',
				'wpbooked_appt_confirmation_email_subject',
				'wpbooked_admin_appointment_email_content',
				'wpbooked_admin_appointment_email_subject',
				'wpbooked_admin_cancellation_email_content',
				'wpbooked_admin_cancellation_email_subject'
			);

			return $plugin_options;
		}

		public function init_settings() {
			$plugin_options = $this->plugin_settings();
			foreach($plugin_options as $option_name) {
				register_setting('wpbooked_plugin-group', $option_name);
			}
		}


		public function wpbooked_phone_numbers($profile_fields) {
			$profile_fields['wpbooked_phone'] = esc_html__('Phone Number','wpbooked');
			return $profile_fields;
		}


		/**********************
		ADD MENUS FUNCTION
		**********************/

		public function add_menu() {
			add_menu_page( esc_html__('WPBooked','wpbooked'), esc_html__('WPBooked','wpbooked'), 'edit_booked_appointments', 'wpbooked-appointments', array($this, 'admin_calendar'), WPBOOKED_PLUGIN_URL . "/assets/images/icon.svg", 58 );
			add_submenu_page('wpbooked-appointments', esc_html__('Pending','wpbooked'), esc_html__('Pending','wpbooked'), 'edit_booked_appointments', "edit.php?post_type=wpbooked_apts" );
			add_submenu_page('wpbooked-appointments', esc_html__('Calendars','wpbooked'), esc_html__('Calendars','wpbooked'), 'manage_booked_options', 'edit-tags.php?taxonomy=wpbooked_custom_calendars');
			add_submenu_page('wpbooked-appointments', esc_html__('Settings','wpbooked'), esc_html__('Settings','wpbooked'), 'edit_booked_appointments', 'wpbooked-settings', array($this, 'plugin_settings_page'));
		}

		public function add_admin_bar_menu() {

			$hide_menu = get_option('wpbooked_hide_admin_bar_menu',false);

			if (!$hide_menu):

				global $wp_admin_bar;

				$wp_admin_bar->add_menu(array('id' => 'wpbooked', 'title' => '<img src="'.esc_url( WPBOOKED_PLUGIN_URL . "/assets/images/icon.svg" ).'" />'.esc_html__('WPBooked','wpbooked'), 'href' => get_admin_url().'admin.php?page=wpbooked-appointments'));
				$wp_admin_bar->add_menu(array('parent' => 'wpbooked', 'title' => esc_html__('Appointments','wpbooked'), 'id' => 'wpbooked-appointments', 'href' => get_admin_url().'admin.php?page=wpbooked-appointments'));
				$wp_admin_bar->add_menu(array('parent' => 'wpbooked', 'title' => esc_html__('Pending','wpbooked'), 'id' => 'wpbooked-pending', 'href' => get_admin_url().'admin.php?page=wpbooked-pending'));
				if (current_user_can('manage_booked_options')):
					$wp_admin_bar->add_menu(array('parent' => 'wpbooked', 'title' => esc_html__('Calendars','wpbooked'), 'id' => 'wpbooked-calendars', 'href' => get_admin_url().'edit-tags.php?taxonomy=wpbooked_custom_calendars'));
				endif;
				$wp_admin_bar->add_menu(array('parent' => 'wpbooked', 'title' => esc_html__('Settings','wpbooked'), 'id' => 'wpbooked-settings', 'href' => get_admin_url().'admin.php?page=wpbooked-settings'));

			endif;

		}

		// Move Taxonomy (custom calendars) to Appointments Menu
		public function wpbooked_tax_menu_correction($parent_file) {
			global $current_screen;
			$taxonomy = $current_screen->taxonomy;
			if ($taxonomy == 'wpbooked_custom_calendars')
				$parent_file = 'wpbooked-appointments';
			return $parent_file;
		}

		// Booked Settings
		public function plugin_settings_page() {
			if(!current_user_can('edit_booked_appointments')) {
				wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'wpbooked'));
			}
			include(sprintf("%s/templates/settings.php", WPBOOKED_PLUGIN_DIR));
		}

		// Booked Pending Appointments List
		public function admin_pending_list() {
			if(!current_user_can('edit_booked_appointments')) {
				wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'wpbooked'));
			}
			
		}

		// Booked Appointment Calendar
		public function admin_calendar() {
			if(!current_user_can('edit_booked_appointments')) {
				wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'wpbooked'));
			}
			include(sprintf("%s/templates/admin-calendar.php", WPBOOKED_PLUGIN_DIR));
		}

		// Add Pending Appointments Bubble
		public function wpbooked_add_pending_appt_bubble() {

			global $submenu;

			$pending = wpbooked_pending_appts_count();

			foreach ( $submenu as $key => $value ) :
				if ( $key == 'booked-appointments' ) :
					if ( $pending ) { $submenu[$key][1][0] .= "&nbsp;<span class='awaiting-mod count-$pending' title='$pending'><span class='pending-count' aria-hidden='true'>$pending</span><span class='comments-in-moderation-text screen-reader-text'>$pending Pending Bookings</span></span>"; }
					return;
				endif;
			endforeach;

		}

		public function wpbooked_no_profile_page_notice() {
			if (current_user_can('manage_booked_options')):

				$booked_booking_type = get_option('wpbooked_booking_type','registered');
				$booked_redirect_type =  get_option('wpbooked_appointment_redirect_type',false);
				$booked_profile_page = wpbooked_get_profile_page();
				$page = (isset($_GET['page']) ? $page = esc_html( $_GET['page'] ) : $page = false);

				if ($booked_booking_type == 'registered' && $booked_redirect_type == 'wpbooked-profile' && !$booked_profile_page && $page != 'wpbooked-welcome'):

					echo '<div class="notice notice-warning" style="line-height:37px; border-left-color:#DB5933;">';
						/* translators: %s: For wpbooked-profile Shortcode. */
						echo sprintf(esc_html__( 'You need to create a page with the %s shortcode. It is required with your current settings.','wpbooked' ),'<code>[wpbooked-profile]</code>').'&nbsp;&nbsp;&nbsp;<a href="'.esc_html(get_admin_url()).'post-new.php?post_type=page">'.esc_html__('Create a Page','wpbooked').'</a>&nbsp;&nbsp;|&nbsp;&nbsp;<a href="'.esc_html(get_admin_url()).'admin.php?page=wpbooked-settings">'.esc_html__('Change Settings','wpbooked').'</a>';
					echo '</div>';

				endif;

			endif;

		}

		public function wpbooked_pending_notice() {

			if (current_user_can('edit_booked_appointments')):

				$pending = wpbooked_pending_appts_count();
				$page = (isset($_GET['page']) ? $page = esc_html( $_GET['page'] ) : $page = false);
				if ($pending && $page != 'wpbooked-pending' && $page != 'wpbooked-welcome'):

					echo '<div class="notice notice-warning" style="line-height:37px">';
						/* translators: %s: Link For Pending Appointment */
						echo sprintf( esc_html(_n( 'There is %s pending appointment.', 'There are %s pending appointments.', $pending, 'wpbooked' )) , $pending ).'&nbsp;&nbsp;<a href="'.esc_html(get_admin_url()).'edit.php?post_type=wpbooked_apts">'.esc_html( _n('View Pending Appointment','View Pending Appointments',$pending,'wpbooked')).' &rarr;</a>'; // phpcs:ignore  WordPress.Security.EscapeOutput.OutputNotEscaped 
					echo '</div>';

				endif;

			endif;

		}

		/**********************
		ADD USER FIELD TO CALENDAR TAXONOMY PAGE
		**********************/
		public function wpbooked_calendars_add_custom_fields($tag) {

			?><div class="form-field">
				<label for="term_meta[notifications_user_id]"><?php esc_html_e('Assign this calendar to','wpbooked'); ?>:</label>
				<select name="term_meta[notifications_user_id]" id="term_meta[notifications_user_id]">
				<option value=""><?php esc_html_e('Default','wpbooked'); ?></option><?php

					$allowed_users = get_users( array( 'role__in' => array( 'administrator', 'wpbooked_booking_agent' ) ) );

					if(!empty($allowed_users)) :
						foreach($allowed_users as $u) :
							$user_id = $u->ID;
							$email = $u->data->user_email;
							$display_name = ( isset( $u->data->display_name ) && $u->data->display_name ? $u->data->display_name . ' (' . $email .')' : $email ); ?>
							<option value="<?php echo esc_attr($email); ?>"><?php echo esc_html($display_name); ?></option><?php
						endforeach;
					endif;

				?></select>
				<p><?php esc_html_e('This will use your setting from the Booked Settings panel by default.','wpbooked'); ?></p>
			</div><?php

		}

		public function wpbooked_calendars_edit_custom_fields($tag) {

			$t_id = $tag->term_id;
			$term_meta = get_option( "taxonomy_$t_id" );
			$selected_value = $term_meta['notifications_user_id'];

			$allowed_users = get_users( array( 'role__in' => array( 'administrator', 'wpbooked_booking_agent' ) ) ); ?>

			<tr class="form-field">
				<th scope="row" valign="top">
					<label for="term_meta[notifications_user_id]"><?php esc_html_e('Assign this calendar to','wpbooked'); ?>:</label>
				</th>
				<td>
					<select name="term_meta[notifications_user_id]" id="term_meta[notifications_user_id]">
						<option value=""><?php esc_html_e('Default','wpbooked'); ?></option>
						<?php if(!empty($allowed_users)) :
							foreach($allowed_users as $u) :
								$user_id = $u->ID;
								$email = $u->data->user_email;
								$display_name = ( isset( $u->data->display_name ) && $u->data->display_name ? $u->data->display_name . ' (' . $email .')' : $email ); ?>
								<option value="<?php echo esc_attr($email); ?>"<?php echo esc_html($selected_value == $email ? ' selected="selected"' : ''); ?>><?php echo esc_html($display_name); ?></option>
							<?php endforeach;

						endif; ?>
					</select><br>
					<span class="description"><?php esc_html_e('This will use your setting from the Booked Settings panel by default.'); ?></span>
				</td>
			</tr><?php
		}

		/**********************
		SAVE USER FIELD FROM CALENDAR TAXONOMY PAGE
		**********************/
		public function wpbooked_save_calendars_custom_fields( $term_id ) {
			if ( isset( $_POST['term_meta'] ) ) {
				$t_id = $term_id;
				$term_meta = get_option( "taxonomy_$t_id" ,[]);
				$cat_keys = array_keys( $_POST['term_meta'] );
				foreach ( $cat_keys as $key ) {
					if ( isset ( $_POST['term_meta'][$key] ) ) {
						$term_meta[$key] = $_POST['term_meta'][$key];
					}
				}
				update_option( "taxonomy_$t_id", $term_meta );
			}
		}

		/**********************
		ADD USER COLUMN FOR APPOINTMENT COUNTS
		**********************/

		public function wpbooked_add_user_columns( $defaults ) {
			$defaults['wpbooked_apts'] = esc_html__('Appointments', 'wpbooked');
			return $defaults;
		}
		public function wpbooked_add_custom_user_columns( $value, $column_name, $id ) {

			if ( $column_name == 'wpbooked_apts' ) {

				$args = array(
					'posts_per_page'   	=> 500,
					'meta_key'   	   	=> '_appointment_timestamp',
					'orderby'			=> 'meta_value_num',
					'order'            	=> 'ASC',
					'meta_query' => array(
						array(
							'key'     => '_appointment_timestamp',
							'value'   => strtotime(date_i18n('Y-m-d H:i:s')),
							'compare' => '>=',
						),
					),
					'author'		   	=> $id,
					'post_type'        	=> 'wpbooked_apts',
					'post_status'      	=> array('publish','future'),
					'suppress_filters'	=> true );

				$appointments = get_posts($args);
				$count = count($appointments);

				$appointments = array_slice($appointments, 0, 5);
				$time_format = get_option('time_format');
				$date_format = get_option('date_format');

				ob_start();

				if ($count){

					echo '<strong>'.esc_html($count).' '.esc_html(_n('Upcoming Appointment','Upcoming Appointments',$count,'wpbooked')).':</strong>';

					echo '<span style="font-size:12px;">';

					foreach($appointments as $appointment):
						$timeslot = get_post_meta($appointment->ID, '_appointment_timeslot',true);
						$timeslot = explode('-',$timeslot);
						$timestamp = get_post_meta($appointment->ID, '_appointment_timestamp',true);
						echo '<br>' . esc_html(date_i18n($date_format,$timestamp)) . ' @ ' . esc_html(date_i18n($time_format,strtotime($timeslot[0]))) . '&ndash;' . esc_html(date_i18n($time_format,strtotime($timeslot[1])));
					endforeach;

					if ($count > 5):
						$diff = $count - 5;
						echo '<br>...'.esc_html__('and','wpbooked').' '.esc_html($diff).' '.esc_html__('more','wpbooked');
					endif;

					echo '</span>';

				}

				return ob_get_clean();

			} else {

				return $value;

			}

		}


		// --------- ADMIN SCRIPTS/STYLES --------- //

		public function admin_scripts() {

			$current_page = (isset($_GET['page']) ? esc_html( $_GET['page'] ) : false);
			$screen = get_current_screen();

			// Gonna need jQuery
			wp_enqueue_script('jquery');

			// For Serializing Arrays
			if ($current_page == 'wpbooked-settings' || $screen->id == 'dashboard'):
				wp_enqueue_script('booked-serialize', WPBOOKED_PLUGIN_URL . '/assets/js/jquery.serialize.js', array(), WPBOOKED_VERSION);
			endif;

			// Load the rest of the stuff!+
			global $post_type;
			if ((in_array($current_page,$this->booked_screens) || $screen->id == 'dashboard') || $post_type == "wpbooked_appointments"):

				wp_enqueue_media();
				wp_enqueue_script('wp-color-picker');
				wp_enqueue_script('jquery-ui');
				wp_enqueue_script('jquery-ui-sortable');
				wp_enqueue_script('jquery-ui-datepicker');
				wp_enqueue_script('spin-js', WPBOOKED_PLUGIN_URL . '/assets/js/spin.min.js', array(), '2.0.1');
				wp_enqueue_script('spin-jquery', WPBOOKED_PLUGIN_URL . '/assets/js/spin.jquery.js', array(), '2.0.1');
				wp_enqueue_script('wpbooked-chosen', WPBOOKED_PLUGIN_URL . '/assets/js/chosen/chosen.jquery.min.js', array(), '1.2.0');
				wp_enqueue_script('wpbooked-fitvids', WPBOOKED_PLUGIN_URL . '/assets/js/fitvids.js', array(), '1.1');
				wp_enqueue_script('wpbooked-tooltipster', WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/js/jquery.tooltipster.min.js', array(), '3.3.0', true);
				wp_register_script('wpbooked-admin', WPBOOKED_PLUGIN_URL . '/assets/js/admin-functions.js', array(), WPBOOKED_VERSION);

				// WPML Compatibility with AJAX calls
				$ajax_url = admin_url( 'admin-ajax.php' );
				$wpml_current_language = apply_filters( 'wpml_current_language', NULL );
				if ( $wpml_current_language ) {
					$ajax_url = add_query_arg( 'wpml_lang', $wpml_current_language, $ajax_url );
				}

				$booked_js_vars = array(
					'ajax_url' => $ajax_url,
					'ajaxRequests' => array(),
					'i18n_slot' => esc_html( _x('Space Available', 'Single Space', 'wpbooked') ),
					'i18n_slots' => esc_html( _x('Spaces Available', 'Multiple Spaces', 'wpbooked') ),
					'i18n_add' => esc_html__('Add Timeslots','wpbooked'),
					'i18n_time_error' => esc_html__('The "End Time" needs to be later than the "Start Time".','wpbooked'),
					'i18n_bulk_add_confirm' => esc_html__('Are you sure you want to add those bulk time slots?','wpbooked'),
					'i18n_all_fields_required' => esc_html__('All fields are required.','wpbooked'),
					'i18n_single_add_confirm' => esc_html__('You are about to add the following time slot(s)','wpbooked'),
					'i18n_to' => esc_html__('to','wpbooked'),
					'i18n_please_wait' => esc_html__('Please wait ...','wpbooked'),
					'i18n_update_appointment' => esc_html__('Update Appointment','wpbooked'),
					'i18n_create_appointment' => esc_html__('Create Appointment','wpbooked'),
					'i18n_all_day' => esc_html__('All day','wpbooked'),
					'i18n_enable' => esc_html__('Enable','wpbooked'),
					'i18n_disable' => esc_html__('Disable','wpbooked'),
					'i18n_change_date' => esc_html__('Change Date','wpbooked'),
					'i18n_choose_customer' => esc_html__('Please choose a customer.','wpbooked'),
					'i18n_fill_out_required_fields' => esc_html__('Please fill out all required fields.','wpbooked'),
					'i18n_confirm_ts_delete' => esc_html__('Are you sure you want to delete this time slot?','wpbooked'),
					'i18n_confirm_cts_delete' => esc_html__('Are you sure you want to delete this custom time slot?','wpbooked'),
					'i18n_confirm_appt_delete' => esc_html__('Are you sure you want to cancel this appointment?','wpbooked'),
					'i18n_clear_timeslots_confirm' => esc_html__('Are you sure you want to delete all of the timeslots for this day?','wpbooked'),
					'i18n_appt_required_fields' => esc_html__('A name, email address and password are required.','wpbooked'),
					'i18n_appt_required_guest_fields' => esc_html__('A name is required.','wpbooked'),
					'i18n_appt_required_guest_fields_surname' => esc_html__('A first and last name are required.','wpbooked'),
					'i18n_appt_required_guest_fields_all' => esc_html__('A first name, last name and email address are required.','wpbooked'),
					'i18n_appt_required_guest_fields_name_email' => esc_html__('A name and an email address are required.','wpbooked'),
					'i18n_confirm_appt_approve' => esc_html__('Are you sure you want to approve this appointment?','wpbooked'),
					'i18n_confirm_appt_approve_all' => esc_html__('Are you sure you want to approve ALL pending appointments?','wpbooked'),
					'i18n_confirm_appt_delete_all' => esc_html__('Are you sure you want to delete ALL pending appointments?','wpbooked'),
					'i18n_confirm_appt_delete_past' => esc_html__('Are you sure you want to delete all PASSED pending appointments?','wpbooked'),
					'nonce' => wp_create_nonce('ajax-admin-nonce')
				);

				wp_localize_script('wpbooked-admin', 'wpbooked_js_vars', $booked_js_vars );
				wp_enqueue_script('wpbooked-admin');

			endif;

		}

		public function admin_styles() {

			$current_page = (isset($_GET['page']) ? esc_html( $_GET['page'] ) : false);
			$screen = get_current_screen();

			if (in_array($current_page,$this->booked_screens) || $screen->id == 'dashboard'):
				wp_enqueue_style('wp-color-picker');
				wp_enqueue_style('wpbooked-tooltipster', 	WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/css/tooltipster.css', array(), '3.3.0');
				wp_enqueue_style('wpbooked-tooltipster-theme', 	WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/css/themes/tooltipster-light.css', array(), '3.3.0');
				wp_enqueue_style('chosen', WPBOOKED_PLUGIN_URL . '/assets/js/chosen/chosen.min.css', array(), '1.2.0');
				wp_enqueue_style('wpbooked-animations', WPBOOKED_PLUGIN_URL . '/assets/css/animations.css', array(), WPBOOKED_VERSION);
				wp_enqueue_style('wpbooked-admin', WPBOOKED_PLUGIN_URL . '/dist/booked-admin.css', array(), WPBOOKED_VERSION);
			endif;

		}


		// --------- FRONT-END SCRIPTS/STYLES --------- //

		public function front_end_scripts() {

			wp_enqueue_script('jquery');
			wp_enqueue_script('jquery-ui');
			wp_enqueue_script('jquery-ui-datepicker');
			wp_register_script('wpbooked-atc', WPBOOKED_PLUGIN_URL . '/assets/js/atc.min.js', array(), '1.6.1', true );
			wp_enqueue_script('wpbooked-spin-js', 	WPBOOKED_PLUGIN_URL . '/assets/js/spin.min.js', array(), '2.0.1', true);
			wp_enqueue_script('wpbooked-spin-jquery', WPBOOKED_PLUGIN_URL . '/assets/js/spin.jquery.js', array(), '2.0.1', true);
			wp_enqueue_script('wpbooked-tooltipster', WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/js/jquery.tooltipster.min.js', array(), '3.3.0', true);
			wp_register_script('wpbooked-functions', WPBOOKED_PLUGIN_URL . '/assets/js/functions.js', array(), WPBOOKED_VERSION, true);

			$booked_redirect_type = get_option('wpbooked_appointment_redirect_type','booked-profile');
			$booked_detect_profile_page = wpbooked_get_profile_page();

			if ($booked_redirect_type == 'booked-profile'):
				$profile_page = ( $booked_detect_profile_page ? $booked_detect_profile_page : false );
			elseif ($booked_redirect_type == 'page'):
				$profile_page = get_option( 'wpbooked_appointment_success_redirect_page',false );
			else:
				$profile_page = false;
			endif;

			$profile_page = ( $profile_page ? esc_url( get_permalink( $profile_page ) ) : false );

			// WPML Compatibility with AJAX calls
			$ajax_url = admin_url( 'admin-ajax.php' );
			$wpml_current_language = apply_filters( 'wpml_current_language', NULL );
			if ( $wpml_current_language ) {
				$ajax_url = add_query_arg( 'wpml_lang', $wpml_current_language, $ajax_url );
			}

			$booked_js_vars = array(
				'ajax_url' => $ajax_url,
				'profilePage' => $profile_page,
				'publicAppointments' => get_option('wpbooked_public_appointments',false),
				'i18n_confirm_appt_delete' => esc_html__('Are you sure you want to cancel this appointment?','wpbooked'),
				'i18n_please_wait' => esc_html__('Please wait ...','wpbooked'),
				'i18n_wrong_username_pass' => esc_html__('Wrong username/password combination.','wpbooked'),
				'i18n_fill_out_required_fields' => esc_html__('Please fill out all required fields.','wpbooked'),
				'i18n_guest_appt_required_fields' => esc_html__('Please enter your name to book an appointment.','wpbooked'),
				'i18n_appt_required_fields' => esc_html__('Please enter your name, your email address and choose a password to book an appointment.','wpbooked'),
				'i18n_appt_required_fields_guest' => esc_html__('Please fill in all "Information" fields.','wpbooked'),
				'i18n_password_reset' => esc_html__('Please check your email for instructions on resetting your password.','wpbooked'),
				'i18n_password_reset_error' => esc_html__('That username or email is not recognized.','wpbooked'),
				'nonce' => wp_create_nonce('ajax-nonce')
			);
			wp_localize_script( 'wpbooked-functions', 'wpbooked_js_vars', $booked_js_vars );
			wp_enqueue_script('wpbooked-functions');

		}

		public static function front_end_styles() {

			wp_enqueue_style('wpbooked-tooltipster', WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/css/tooltipster.css', array(), '3.3.0');
			wp_enqueue_style('wpbooked-tooltipster-theme', WPBOOKED_PLUGIN_URL . '/assets/js/tooltipster/css/themes/tooltipster-light.css', array(), '3.3.0');
			wp_enqueue_style('wpbooked-animations', WPBOOKED_PLUGIN_URL . '/assets/css/animations.css', array(), WPBOOKED_VERSION);
			wp_enqueue_style('wpbooked-css', WPBOOKED_PLUGIN_URL . '/dist/booked.css', array(), WPBOOKED_VERSION);

			if ( defined('NECTAR_THEME_NAME') && NECTAR_THEME_NAME == 'salient' ):
				wp_enqueue_style('booked-salient-overrides', WPBOOKED_PLUGIN_URL . '/assets/css/theme-specific/salient.css', array(), WPBOOKED_VERSION);
			endif;
			
			if (!isset($_GET['print'])):
				$colors_pattern_file = WPBOOKED_PLUGIN_DIR . '/assets/css/color-theme.php';
				if ( !file_exists($colors_pattern_file) ) {
					return;
				}
			
				ob_start();
				include(esc_attr($colors_pattern_file));
				$booked_color_css = ob_get_clean();
				$compressed_booked_color_css = wpbooked_compress_css( $booked_color_css );
				
				wp_add_inline_style( 'wpbooked-css', $compressed_booked_color_css );
			
			endif;

		}

		public static function wpbooked_user_roles_filter( $booked_user_roles ) {
			return $booked_user_roles;
		}

	} // END class wpbooked_plugin
} // END if(!class_exists('wpbooked_plugin'))

if(class_exists('wpbooked_plugin')) {

	// Add the "Booking Agent" User Role
	$booking_agent = add_role(
	    'wpbooked_booking_agent',
	    esc_html__( 'WP Booking Agent','wpbooked' ),
	    array(
	        'read' => true,
	    )
	);

	add_action( 'wp_roles_init', static function ( \WP_Roles $roles ) {
		$roles->roles['wpbooked_booking_agent']['name'] =  esc_html__( 'WP Booking Agent','wpbooked' );;
		$roles->role_names['wpbooked_booking_agent'] =  esc_html__( 'WP Booking Agent','wpbooked' );;
	} );

	// Add Capabilities to User Roles (the below array can be filtered to include more or exclude any of the defaults)
	$booked_user_roles = apply_filters( 'wpbooked_user_roles', array('administrator','wpbooked_booking_agent') );

	foreach($booked_user_roles as $role_name):
		$role_caps = get_role($role_name);
		$role_caps->add_cap( 'edit_booked_appointments' );
	endforeach;

	$booked_admin_caps = get_role( 'administrator' );
	$booked_admin_caps->add_cap('manage_booked_options');

	// Activation Hook
	register_activation_hook( __FILE__, array('wpbooked_plugin', 'activate'));
	register_activation_hook( __FILE__, 'FortAwesome\FontAwesome_Loader::initialize' );
	register_deactivation_hook( __FILE__, 'FortAwesome\FontAwesome_Loader::maybe_deactivate' );
	
	add_action(
		'font_awesome_preferences',
		function() {
			fa()->register(
				array(
					'name' => 'Booked',
					'version' => [
						[ '6.1.2', '>=' ]
					]
				)
			);
		}
	);

	// Initiate the Booked Class
	$wpbooked_plugin = new wpbooked_plugin();

	// Add a link to the settings page onto the plugin page
	if(isset($wpbooked_plugin)) {

		// Add the settings link to the plugins page
		function wpbooked_custom_links($links) {

			$custom_links[] = '<a href="admin.php?page=wpbooked-settings">'.esc_html__('Settings','wpbooked').'</a>';			

			foreach($custom_links as $custom_link):
				array_unshift($links, $custom_link);
			endforeach;

			return $links;

		}

		$plugin = plugin_basename(__FILE__);
		add_filter("plugin_action_links_$plugin", 'wpbooked_custom_links');

		// Load the Front-End Styles and Color Settings
		add_action('wp_enqueue_scripts', array('wpbooked_plugin', 'front_end_styles'));

	}
}

// Localization
function wpbooked_local_init(){
	$domain = 'wpbooked';
	$locale = apply_filters('plugin_locale', get_locale(), $domain);
	load_textdomain($domain, WP_LANG_DIR.'/booked/'.$domain.'-'.$locale.'.mo');
	load_plugin_textdomain($domain, FALSE, dirname(plugin_basename(__FILE__)).'/languages/');
}
add_action('after_setup_theme', 'wpbooked_local_init');
